<?php

namespace Arts\PluginsUpdater;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use \Arts\PluginsUpdater\Managers\BundledPlugin;
use \Arts\PluginsUpdater\Managers\CurrentPlugin;

/**
 * Class Plugin
 *
 * Main plugin class that initializes the plugin and adds managers.
 *
 * @package Arts\PluginsUpdater
 */
class Plugin {
	/**
	 * The theme ID associated with the plugin.
	 *
	 * @var string
	 * @example Example `my-theme-slug`
	 */
	private $theme_id;

	/**
	 * Managers for the plugin.
	 *
	 * @var Object
	 */
	private $managers;

	public function __construct( $args = array(), $strings = array() ) {
		$this->managers                  = new \stdClass();
		$this->managers->bundled_plugins = new \stdClass();
		$this->init( $args, $strings );
	}

	/**
	 * Initializes the plugin by adding managers, filters, and actions.
	 *
	 * @param array $args Arguments to pass to the manager classes.
	 * @param array $strings Strings to pass to the manager classes.
	 *
	 * @return Plugin Returns the current instance for method chaining.
	 */
	private function init( $args, $strings ) {
		$this->theme_id = $args['theme_id'];

		$this
		->add_managers( $args, $strings )
		->init_managers()
		->add_filters( $args )
		->add_actions( $args );

		return $this;
	}

	/**
	 * Adds manager instances to the managers property.
	 *
	 * @param array $args Arguments to pass to the manager classes.
	 * @param array $strings Strings to pass to the manager classes.
	 *
	 * @return Plugin Returns the current instance for method chaining.
	 */
	private function add_managers( $args, $strings ) {
		if ( isset( $args['current_plugin'] ) && is_array( $args['current_plugin'] ) ) {
			$current_plugin_manager_class = CurrentPlugin::class;
			$current_plugin_args          = $args['current_plugin'];

			$this->managers->current_plugin = $this->get_manager_instance( $current_plugin_manager_class, $current_plugin_args, $strings );
		}

		if ( is_array( $args['bundled_plugins'] ) && ! empty( $args['bundled_plugins'] ) ) {
			$bundled_plugin_manager_class = BundledPlugin::class;

			foreach ( $args['bundled_plugins'] as $key => $class_args ) {
				$this->managers->bundled_plugins->$key = $this->get_manager_instance( $bundled_plugin_manager_class, $class_args, $strings );
			}
		}

		return $this;
	}

	/**
	 * Initialize all manager classes by calling their init method if it exists.
	 *
	 * @return Plugin Returns the current instance for method chaining.
	 */
	private function init_managers() {
		$all_managers = array_merge(
			(array) $this->managers,
			(array) $this->managers->bundled_plugins
		);

		foreach ( $all_managers as $manager ) {
			if ( method_exists( $manager, 'init' ) ) {
					$manager->init( $all_managers );
			}
		}

		return $this;
	}

	/**
	 * Helper method to instantiate a manager class.
	 *
	 * @param string $class The manager class to instantiate.
	 * @param array  $args Arguments to pass to the manager class.
	 * @param array  $strings Strings to pass to the manager class.
	 *
	 * @return object The instantiated manager class.
	 */
	private function get_manager_instance( $class, $args, $strings ) {
		try {
			$reflection = new \ReflectionClass( $class );
			return $reflection->newInstanceArgs( array( $this->theme_id, $args, $strings ) );
		} catch ( \ReflectionException $e ) {
			return new $class();
		}
	}

	/**
	 * Adds WordPress actions for the plugin.
	 *
	 * @param array $args Arguments to pass to the action hooks.
	 *
	 * @return Plugin Returns the current instance for method chaining.
	 */
	private function add_actions( $args = array() ) {
		if ( empty( $this->theme_id ) ) {
			return $this;
		}

		// Clear the plugins cache when the license key or license key status changes.
		add_action( "update_option_{$this->theme_id}_license_key", array( $this, 'clear_update_data' ), 10, 2 );
		add_action( "update_option_{$this->theme_id}_license_key_status", array( $this, 'clear_update_data' ), 10, 2 );

		return $this;
	}
	/**
	 * Adds WordPress filters for the plugin.
	 *
	 * @return Plugin Returns the current instance for method chaining.
	 */
	private function add_filters( $args = array() ) {
		return $this;
	}

	/**
	 * Clears the plugins cache to ensure that update data is refreshed.
	 *
	 * This method calls `wp_clean_plugins_cache` to clear the plugins cache, forcing WordPress to fetch
	 * fresh update data. This can be useful when the license key status changes or when an update is
	 * manually triggered.
	 *
	 * @return void
	 */
	public function clear_update_data() {
		wp_clean_plugins_cache( true );
	}
}
