<?php

namespace Arts\PluginsUpdater\Managers;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class CurrentPlugin
 *
 * Manages the updates for the current plugin.
 *
 * @package Arts\PluginsUpdater\Managers
 */
class CurrentPlugin extends BaseManager {
	/**
	 * Icons to appear on the `update-core.php` page.
	 *
	 * @var array
	 */
	private $icons;

	/**
	 * Banners to appear on the `update-core.php` page.
	 *
	 * @var array
	 */
	private $banners;

	/**
	 * The update URI for the plugin.
	 *
	 * This is used to filter the update transient by comparing the "Update URI" field in the plugin data.
	 *
	 * @var string
	 */
	private $update_uri;

	/**
	 * The plugin slug, derived from the plugin ID.
	 *
	 * @var string
	 */
	private $plugin_slug;

	public function init( $managers ) {
		$this->plugin_id  = $this->args['plugin_id'];
		$this->endpoint   = $this->args['endpoint'];
		$this->method     = $this->args['method'];
		$this->icons      = $this->args['icons'];
		$this->banners    = $this->args['banners'];
		$this->update_uri = $this->args['update_uri'];

		$plugin_file = trailingslashit( WP_PLUGIN_DIR ) . $this->plugin_id;

		// Robust slug detection
		if ( file_exists( $plugin_file ) ) {
			$this->plugin_slug = dirname( plugin_basename( $plugin_file ) );
		} else {
			// Fallback if the file doesn't exist
			$parts             = explode( '/', $this->plugin_id );
			$this->plugin_slug = $parts[0];
		}

		$this->add_managers( $managers );
		$this->add_actions();
		$this->add_filters();
	}

	protected function add_actions() {
		if ( ! empty( $this->plugin_id ) ) {
			add_action( "in_plugin_update_message-{$this->plugin_id}", array( $this, 'modify_update_message' ), 10, 2 );
		}

		return $this;
	}

	protected function add_filters() {
		// Set the remote data for the current plugin.
		add_filter( 'plugins_api', array( $this, 'set_remote_data' ), 99, 3 );

		if ( ! empty( $this->update_uri ) ) {
			// Check for the current plugin updates.
			add_filter( "update_plugins_{$this->update_uri}", array( $this, 'check_update' ), 99, 4 );
		}
	}

	/**
	 * Sets the remote data for the current plugin.
	 *
	 * This method is used to modify the plugin information object with data fetched from a remote source.
	 *
	 * @param mixed  $res    The current plugin information object.
	 * @param string $action The action being performed. Should be 'plugin_information'.
	 * @param object $args   The arguments passed to the action. Should contain the plugin slug.
	 *
	 * @return mixed The modified plugin information object or the original object if conditions are not met.
	 */
	public function set_remote_data( $res, $action, $args ) {
		// The action should be 'plugin_information'.
		if ( $action !== 'plugin_information' ) {
			return $res;
		}

		// The plugin slug should match the current plugin slug.
		if ( ! isset( $args->slug ) || $args->slug !== $this->plugin_slug ) {
			return $res;
		}

		$remote_plugin_data = $this->fetch_remote_data( $this->license_key, $this->endpoint, $this->method );

		// Remote data is not available.
		if ( is_wp_error( $remote_plugin_data ) ) {
			return $res;
		}

		// Modify the plugin information object with the remote data.
		$res                 = new \stdClass();
		$res->name           = $remote_plugin_data->name;
		$res->slug           = $remote_plugin_data->slug;
		$res->version        = $remote_plugin_data->version;
		$res->tested         = $remote_plugin_data->tested;
		$res->requires       = $remote_plugin_data->requires;
		$res->author         = $remote_plugin_data->author;
		$res->author_profile = $remote_plugin_data->author_profile;
		$res->donate_link    = $remote_plugin_data->donate_link;
		$res->homepage       = $remote_plugin_data->homepage;
		$res->download_link  = $remote_plugin_data->download_url;
		$res->trunk          = $remote_plugin_data->download_url;
		$res->requires_php   = $remote_plugin_data->requires_php;
		$res->last_updated   = $remote_plugin_data->last_updated;
		$res->sections       = $remote_plugin_data->sections;
		$res->rating         = $remote_plugin_data->rating;
		$res->num_ratings    = $remote_plugin_data->num_ratings;

		// Add banners to the update object.
		if ( is_array( $this->banners ) && ! empty( $this->banners ) ) {
			$res->banners = $this->banners;
		}

		return $res;
	}

	/**
	 * Checks for plugin updates by comparing the local version with the remote version.
	 *
	 * This method fetches remote plugin data using the license key and endpoint, and compares the
	 * remote version with the local version. If an update is available, it modifies the update object
	 * with the new version and download URL.
	 *
	 * @param object $update The update object.
	 * @param array  $plugin_data The plugin data array.
	 * @param string $plugin_file The plugin file path.
	 * @param array  $locales The locales array.
	 *
	 * @return object The modified update object.
	 */
	public function check_update( $update, $plugin_data, $plugin_file, $locales ) {
		// The plugin file should match the current plugin ID.
		if ( $plugin_file !== $this->plugin_id || ! empty( $update ) ) {
			return $update;
		}

		$remote_plugin_data = $this->fetch_remote_data( $this->license_key, $this->endpoint, $this->method );

		// Remote data is not available.
		if ( is_wp_error( $remote_plugin_data ) ) {
			return $update;
		}

		$remote_version = isset( $remote_plugin_data->version ) ? $remote_plugin_data->version : '';

		// No update needed since the remote version is the same or lower than the local version.
		if ( ! version_compare( $plugin_data['Version'], $remote_version, '<' ) ) {
			return $update;
		}

		// Modify the update object with the new version and download URL.
		$remote_plugin_data->new_version = $remote_plugin_data->version;
		$remote_plugin_data->package     = $remote_plugin_data->download_url;

		// Add icons to the update object.
		if ( is_array( $this->icons ) && ! empty( $this->icons ) ) {
			$remote_plugin_data->icons = $this->icons;
		}

		// Add banners to the update object.
		if ( is_array( $this->banners ) && ! empty( $this->banners ) ) {
			$remote_plugin_data->banners = $this->banners;
		}

		return $remote_plugin_data;
	}

	/**
	 * Modifies the update notification for the current plugin.
	 *
	 * @param array $plugin_data {
	 *  An array of plugin data.
	 *
	 *  @type string $package       The package URL for the plugin update.
	 *  @type string $new_version   The new version of the plugin.
	 *  @type string $Version       The current version of the plugin.
	 *  @type string $purchase_url  The URL to purchase the plugin.
	 *  @type string $message       The message template for the update notice.
	 * }
	 *
	 * @return void
	 */
	public function modify_update_message( $plugin_data ) {
		$no_package      = ! isset( $plugin_data['package'] ) || empty( $plugin_data['package'] );
		$new_version     = isset( $plugin_data['new_version'] ) ? $plugin_data['new_version'] : '';
		$current_version = isset( $plugin_data['Version'] ) ? $plugin_data['Version'] : '';

		if ( $no_package && version_compare( $current_version, $new_version, '<' ) ) {
			$purchase_url               = isset( $plugin_data['purchase_url'] ) ? $plugin_data['purchase_url'] : '';
			$message_template           = isset( $plugin_data['message'] ) ? $plugin_data['message'] : '';
			$message_theme_license_link = sprintf( '<a href="%1$s">%2$s</a>', esc_url( $this->strings['license-page-url'] ), esc_html( $this->strings['license-page-text'] ) );
			$message_purchase_link      = sprintf( '<a href="%1$s" target="_blank" rel="nofollow">%2$s</a>', esc_url( $purchase_url ), esc_html( $this->strings['license-purchase-url-text'] ) );

			$output = sprintf( $message_template, $message_theme_license_link, $message_purchase_link );
			$output = '<br>' . $output;

			echo wp_kses_post( $output );
		}
	}
}
