<?php

namespace Arts\PluginsUpdater\Managers;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class BundledPlugin
 *
 * Manages the updates for theme bundled plugin.
 *
 * @package Arts\PluginsUpdater\Managers
 */
class BundledPlugin extends BaseManager {
	public function init( $managers ) {
		$this->plugin_id = $this->args['plugin_id'];
		$this->endpoint  = $this->args['endpoint'];
		$this->method    = $this->args['method'];

		$this->add_managers( $managers );
		$this->add_actions();
		$this->add_filters();
	}

	protected function add_actions() {
		// Remove the update message for the bundled plugin if the theme license key is valid.
		if ( $this->license_key && $this->license_key_status === 'valid' ) {
			add_action( 'admin_init', array( $this, 'remove_update_message' ) );
		}

		return $this;
	}

	protected function add_filters() {
		// Modify the plugins transient to include the download URL for the bundled plugin.
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'modify_plugins_transient' ), 99, 1 );

		return $this;
	}

	/**
	 * Modifies the plugins transient to include the download URL for the bundled plugin.
	 *
	 * This method checks if the plugin is in the transient response and if its package URL is empty.
	 * If so, it fetches the remote plugin data using the license key and endpoint, and updates the
	 * transient response with the download URL.
	 *
	 * @param object $transient The plugins transient object.
	 * @return object The modified plugins transient object.
	 */
	public function modify_plugins_transient( $transient ) {
		// Check if $transient is an object and has required properties
		if ( ! is_object( $transient ) || ! isset( $transient->response ) || ! is_array( $transient->response ) ) {
			return $transient;
		}

		if ( isset( $transient->response[ $this->plugin_id ] ) && empty( $transient->response[ $this->plugin_id ]->package ) ) {
			$remote_plugin_data = $this->fetch_remote_data( $this->license_key, $this->endpoint, $this->method );

			if ( ! is_wp_error( $remote_plugin_data ) && ! empty( $remote_plugin_data->download_url ) ) {
				$transient->response[ $this->plugin_id ]->package = $remote_plugin_data->download_url;
				$this->remove_update_message();
			}
		}

		return $transient;
	}

	/**
	 * Removes all update messages for the specified plugin.
	 *
	 * @return void
	 */
	public function remove_update_message() {
		remove_all_actions( 'in_plugin_update_message-' . $this->plugin_id );
	}
}
