<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Arrigo_Elementor_Step_Widget extends Arr_Widget_Base {
	protected static $_instance;

	public function get_name() {
		return 'arrigo-widget-step';
	}

	public function get_title() {
		return _x( 'Step', 'Elementor Widget Title', 'arrigo' );
	}

	public function get_icon() {
		return 'eicon-plug';
	}

	public function get_categories() {
		return array( 'arrigo-static' );
	}

	public function wpml_widgets_to_translate_filter( $widgets ) {
		$name  = $this->get_name();
		$title = $this->get_title();

		$widgets[ $name ] = array(
			'conditions' => array( 'widgetType' => $name ),
			'fields'     => array(
				array(
					'field'       => 'counter',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, _x( 'Counter', 'Elementor Widget', 'arrigo' ) ),
					'editor_type' => 'LINE',
				),
				array(
					'field'       => 'heading',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, _x( 'Heading', 'Elementor Widget', 'arrigo' ) ),
					'editor_type' => 'LINE',
				),
				array(
					'field'       => 'description',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, _x( 'Description', 'Elementor Widget', 'arrigo' ) ),
					'editor_type' => 'AREA',
				),
			),
		);

		return $widgets;
	}

	public function add_wpml_support() {
		add_filter( 'wpml_elementor_widgets_to_translate', array( $this, 'wpml_widgets_to_translate_filter' ) );
	}

	protected function register_controls() {
		/**
		 * Section Content
		 */
		$this->start_controls_section(
			'content_section',
			array(
				'label' => _x( 'Content', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			)
		);

		/**
		 * Counter
		 */
		$this->add_control(
			'counter',
			array(
				'label'   => _x( 'Counter', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => '01',
			)
		);

		/**
		 * Heading
		 */
		$this->add_control(
			'heading',
			array(
				'label'       => _x( 'Heading', 'Elementor Widget', 'arrigo' ),
				'label_block' => true,
				'type'        => \Elementor\Controls_Manager::TEXT,
				'default'     => _x( 'Edit heading...', 'Elementor Widget', 'arrigo' ),
			)
		);

		/**
		 * Description
		 */
		$this->add_control(
			'description',
			array(
				'label'   => _x( 'Description', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::TEXTAREA,
				'default' => _x( 'Edit description...', 'Elementor Widget', 'arrigo' ),
			)
		);

		/**
		 * Show Headline
		 */
		$this->add_control(
			'show_headline',
			array(
				'label'   => _x( 'Show decoration line', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();

		/**
		 * Section Image
		 */
		$this->start_controls_section(
			'image_section',
			array(
				'label' => _x( 'Image', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			)
		);

		/**
		 * Image Src
		 */
		$this->add_control(
			'image',
			array(
				'label'   => _x( 'Choose Image', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::MEDIA,
				'default' => array(
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				),
			)
		);

		/**
		 * Image Fit
		 */
		$this->add_control(
			'image_fit',
			array(
				'label'     => _x( 'Image Fit Mode', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => array(
					'contain' => array(
						'title' => _x( 'Contain', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-image-before-after',
					),
					'cover'   => array(
						'title' => _x( 'Cover', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-image',
					),
				),
				'default'   => 'cover',
				'toggle'    => false,
				'condition' => array(
					'image!' => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		/**
		 * Image Parallax
		 */
		$this->add_control(
			'image_parallax',
			array(
				'label'     => _x( 'Enable parallax', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => array(
					'image!' => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		/**
		 * Image Parallax Speed
		 */
		$this->add_control(
			'image_parallax_speed',
			array(
				'label'     => _x( 'Parallax Speed', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => array(
					'factor' => array(
						'min'  => -1.0,
						'max'  => 2.0,
						'step' => 0.01,
					),
				),
				'default'   => array(
					'unit' => 'factor',
					'size' => 0.9,
				),
				'condition' => array(
					'image_parallax' => 'yes',
					'image!'         => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		/**
		 * Overlay
		 */
		$this->add_responsive_control(
			'overlay',
			array(
				'label'          => _x( 'Image Overlay', 'Elementor Widget', 'arrigo' ),
				'label_block'    => true,
				'type'           => \Elementor\Controls_Manager::SELECT,
				'options'        => array(
					'initial' => _x( 'None', 'Elementor Widget', 'arrigo' ),
					'#181818' => _x( 'Dark', 'Elementor Widget', 'arrigo' ),
					'#ffffff' => _x( 'Light', 'Elementor Widget', 'arrigo' ),
				),
				'default'        => 'initial',
				'default_tablet' => '#ffffff',
				'default_mobile' => '#ffffff',
				'selectors'      => array(
					'{{WRAPPER}} .overlay' => 'background-color: {{overlay}};',
				),
				'condition'      => array(
					'image!' => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		/**
		 * Overlay Opacity
		 */
		$this->add_control(
			'overlay_opacity',
			array(
				'label'     => __( 'Image Overlay Opacity', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'default'   => array(
					'size' => .8,
				),
				'range'     => array(
					'px' => array(
						'max'  => 1,
						'step' => 0.01,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .overlay' => 'opacity: {{SIZE}};',
				),
				'condition' => array(
					'image!' => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Section Layout
		 */
		$this->start_controls_section(
			'layout_section',
			array(
				'label' => _x( 'Layout', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_LAYOUT,
			)
		);

		/**
		 * Layout
		 */
		$this->add_control(
			'layout',
			array(
				'label'   => _x( 'Image Position', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::CHOOSE,
				'options' => array(
					'offset-lg-6' => array(
						'title' => _x( 'Image Left', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-h-align-left',
					),
					''            => array(
						'title' => _x( 'Image Right', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-h-align-right',
					),
				),
				'default' => 'offset-lg-6',
				'toggle'  => false,
			)
		);

		$this->end_controls_section();

		/**
		 * Section Style
		 */
		$this->start_controls_section(
			'style_section',
			array(
				'label' => _x( 'Style', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			)
		);

		/**
		 * Inverse Text Color
		 */
		$this->add_control(
			'inverse_text_color',
			array(
				'label'        => _x( 'Inverse Text Color', 'Elementor Widget', 'arrigo' ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'default'      => 'no',
				'return_value' => 'color-white',
			)
		);

		/**
		 * Content Paddings
		 */

		$this->add_responsive_control(
			'content_paddings',
			array(
				'label'      => _x( 'Content Top/Bottom Paddings', 'Elementor Widget', 'arrigo' ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'default'    => array(
					'size' => 150,
					'unit' => 'px',
				),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 1440,
					),
					'vh' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'size_units' => array( 'px', 'vh' ),
				'selectors'  => array(
					'{{WRAPPER}} .section-steps__content' => 'padding-top: {{SIZE}}{{UNIT}}; padding-bottom: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Section Animation
		 */
		$this->start_controls_section(
			'animation_section',
			array(
				'label' => _x( 'Animation', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			)
		);

		/**
		 * OS Animation
		 */
		$this->add_control(
			'enable_animation',
			array(
				'label'        => _x( 'Enable on-scroll animation', 'Elementor Widget', 'arrigo' ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'return_value' => 'data-os-animation="data-os-animation"',
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'counter' );
		$this->add_inline_editing_attributes( 'heading' );
		$this->add_inline_editing_attributes( 'description' );

		$this->add_render_attribute( 'section', 'class', array( 'section', 'section-steps' ) );
		$this->add_render_attribute( 'image', 'class', array( 'col-lg-6', 'section-steps__bg' ) );
		$this->add_render_attribute( 'content', 'class', array( 'col-lg-6', 'section-steps__content', $settings['layout'] ) );

		if ( $settings['layout'] != 'offset-lg-6' ) {
			$this->add_render_attribute( 'image', 'class', 'offset-lg-6' );
		}

		if ( $settings['inverse_text_color'] ) {
			$this->add_render_attribute( 'section', 'class', $settings['inverse_text_color'] );
		}

		if ( $settings['image_parallax'] ) {
			$this->add_render_attribute(
				'image',
				array(
					'class'         => 'jarallax',
					'data-speed'    => is_array( $settings['image_parallax_speed'] ) ? $settings['image_parallax_speed']['size'] : 0,
					'data-img-size' => $settings['image_fit'],
				)
			);
		}

		if ( $settings['enable_animation'] ) {
			$this->add_render_attribute( 'section', 'data-os-animation' );
		}
		?>
		<div <?php $this->print_render_attribute_string( 'section' ); ?>>
			<?php if ( ! empty( $settings['image']['url'] ) ) : ?>
				<div <?php $this->print_render_attribute_string( 'image' ); ?>>
					<?php
						arr_the_lazy_image(
							array(
								'id'    => $settings['image']['id'],
								'type'  => 'image',
								'class' => array(
									'wrapper' => false,
									'image'   => array( 'jarallax-img', 'of-' . $settings['image_fit'] ),
								),
							)
						);
					?>
					<div class="overlay"></div>
				</div>
			<?php endif; ?>
			<div class="container">
				<div class="row align-items-center">
					<div <?php $this->print_render_attribute_string( 'content' ); ?>>
						<?php if ( ! empty( $settings['counter'] ) ) : ?>
							<div class="section-steps__number">
								<span <?php $this->print_render_attribute_string( 'counter' ); ?>><?php echo $settings['counter']; ?></span>
							</div>
						<?php endif; ?>
						<?php if ( ! empty( $settings['heading'] ) ) : ?>
							<h2 <?php $this->print_render_attribute_string( 'heading' ); ?>><?php echo $settings['heading']; ?></h2>
						<?php endif; ?>
						<?php if ( ! empty( $settings['description'] ) ) : ?>
							<p <?php $this->print_render_attribute_string( 'description' ); ?>><?php echo $settings['description']; ?></p>
						<?php endif; ?>
						<?php if ( $settings['show_headline'] ) : ?>
							<div class="section-steps__headline"></div>
						<?php endif; ?>
					</div>
				</div>
			</div>
		</div>
		<?php
	}

	protected function content_template() {
		?>
			<#
				view.addInlineEditingAttributes( 'counter' );
				view.addInlineEditingAttributes( 'heading' );
				view.addInlineEditingAttributes( 'description' );

				view.addRenderAttribute( 'section', 'class', [ 'section', 'section-steps' ] );
				view.addRenderAttribute( 'image', 'class', [ 'col-lg-6', 'section-steps__bg' ] );
				view.addRenderAttribute( 'content', 'class', [ 'col-lg-6', 'section-steps__content', settings.layout ] );
				view.addRenderAttribute( 'overlay', 'class', [ 'overlay', 'd-lg-none', settings.overlay ] );

				if ( settings.layout != 'offset-lg-6' ) {
					view.addRenderAttribute( 'image', 'class', 'offset-lg-6' );
				}

				if ( settings.inverse_text_color ) {
					view.addRenderAttribute( 'section', 'class', settings.inverse_text_color );
				}

				if (settings.image_parallax) {
					view.addRenderAttribute(
						'image', {
							'class': 'jarallax',
							'data-speed': settings.image_parallax_speed.size,
							'data-img-size': settings.image_fit
						}
					);
					view.addRenderAttribute( 'img', 'class', 'jarallax-img' );
				}
			#>
			<div {{{ view.getRenderAttributeString( 'section' ) }}}>
				<# if (settings.image.url) { #>
					<div {{{ view.getRenderAttributeString( 'image' ) }}}>
						<#
							view.addRenderAttribute(
								'img', {
									'class': 'of-' + settings.image_fit,
									'src': settings.image.url,
									'alt': ''
								}
							);
						#>
						<img {{{ view.getRenderAttributeString( 'img' ) }}}/>
						<div class="overlay"></div>
					</div>
				<# } #>
				<div class="container">
					<div class="row align-items-center">
						<div {{{ view.getRenderAttributeString( 'content' ) }}}>
							<# if (settings.counter) { #>
								<div class="section-steps__number">
									<span {{{ view.getRenderAttributeString( 'counter' ) }}}>{{{ settings.counter }}}</span>
								</div>
							<# } #>
							<# if (settings.heading) { #>
								<h2 {{{ view.getRenderAttributeString( 'heading' ) }}}>{{{ settings.heading }}}</h2>
							<# } #>
							<# if (settings.description) { #>
								<p {{{ view.getRenderAttributeString( 'description' ) }}}>{{{ settings.description }}}</p>
							<# } #>
							<# if (settings.show_headline) { #>
								<div class="section-steps__headline"></div>
							<# } #>
						</div>
					</div>
				</div>
			</div>
		<?php
	}
}
