<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Arrigo_Elementor_Images_Slider_Widget extends Arr_Widget_Base {
	protected static $_instance;

	public function get_name() {
		return 'arrigo-widget-images-slider';
	}

	public function get_title() {
		return _x( 'Images Slider', 'Elementor Widget Title', 'arrigo' );
	}

	public function get_icon() {
		return 'eicon-plug';
	}

	public function get_categories() {
		return array( 'arrigo-static' );
	}

	public function get_style_depends() {
		return array( 'swiper' );
	}

	private function check_featured_image() {
		return has_post_thumbnail();
	}

	protected function register_controls() {
		/**
		 * Section Content
		 */
		$this->start_controls_section(
			'content_section',
			array(
				'label' => _x( 'Images', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			)
		);

		/**
		 * Check for Featured Image
		 */
		$this->add_control(
			'has_featured_image',
			array(
				'type'    => \Elementor\Controls_Manager::HIDDEN,
				'default' => $this->check_featured_image(),
			)
		);

		/**
		 * Gallery
		 */
		$this->add_control(
			'images_gallery',
			array(
				'type'    => \Elementor\Controls_Manager::GALLERY,
				'default' => array(),
			)
		);

		/**
		 * Cover Image To Slider
		 */
		$this->add_control(
			'cover_to_slider',
			array(
				'label'     => _x( 'Use Featured Image as first slide', 'Elementor Widget Section', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => array(
					'images_gallery!' => '',
				),
			)
		);

		/**
		 * No featured Image
		 */
		$this->add_control(
			'no_featured_image_info',
			array(
				'type'            => \Elementor\Controls_Manager::RAW_HTML,
				'raw'             => sprintf(
					'%1$s',
					_x( 'Please make sure you have Featured Image for set for the current page in WordPress post editor.', 'Elementor Widget', 'arrigo' )
				),
				'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
				'condition'       => array(
					'images_gallery!'    => '',
					'cover_to_slider'    => 'yes',
					'has_featured_image' => false,
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Section Slider
		 */
		$this->start_controls_section(
			'slider_section',
			array(
				'label' => _x( 'Slider', 'Elementor Widget', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_SETTINGS,
			)
		);

		/**
		 * Slides Per Screen
		 */
		$this->add_responsive_control(
			'slides_per_view',
			array(
				'label'              => _x( 'Slides Per Screen', 'Elementor Widget', 'arrigo' ),
				'type'               => \Elementor\Controls_Manager::SLIDER,
				'range'              => array(
					'number' => array(
						'min'  => 1,
						'max'  => 4,
						'step' => 1,
					),
				),
				'devices'            => array( 'desktop', 'tablet', 'mobile' ),
				'default'            => array(
					'size' => 1,
					'unit' => 'number',
				),
				'desktop_default'    => array(
					'size' => 1,
					'unit' => 'number',
				),
				'tablet_default'     => array(
					'size' => 1,
					'unit' => 'number',
				),
				'mobile_default'     => array(
					'size' => 1,
					'unit' => 'number',
				),
				'frontend_available' => true,
			)
		);

		/**
		 * Centered Slides
		 */
		$this->add_responsive_control(
			'centered_slides',
			array(
				'label'              => _x( 'Centered Slides', 'Elementor Widget', 'arrigo' ),
				'label_block'        => true,
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'frontend_available' => true,
			)
		);

		/**
		 * Space Between Slides
		 */
		$this->add_responsive_control(
			'space_between',
			array(
				'label'              => _x( 'Space Between Slides', 'Elementor Widget', 'arrigo' ),
				'type'               => \Elementor\Controls_Manager::SLIDER,
				'range'              => array(
					'px' => array(
						'min'  => 0,
						'max'  => 160,
						'step' => 1,
					),
				),
				'devices'            => array( 'desktop', 'tablet', 'mobile' ),
				'default'            => array(
					'size' => 0,
					'unit' => 'px',
				),
				'desktop_default'    => array(
					'size' => 0,
					'unit' => 'px',
				),
				'tablet_default'     => array(
					'size' => 0,
					'unit' => 'px',
				),
				'mobile_default'     => array(
					'size' => 0,
					'unit' => 'px',
				),
				'frontend_available' => true,
			)
		);

		/**
		 * Navigation
		 */
		$this->add_control(
			'enable_navigation',
			array(
				'label'   => _x( 'Enable Navigation Arrows', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		/**
		 * Progress Bar
		 */
		$this->add_control(
			'enable_progress',
			array(
				'label'   => _x( 'Enable Progress Bar', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		/**
		 * Autoplay
		 */
		$this->add_control(
			'enable_autoplay',
			array(
				'label'   => _x( 'Autoplay', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		/**
		 * Autoplay Delay
		 */
		$this->add_control(
			'autoplay_delay',
			array(
				'label'     => _x( 'Autoplay Delay (ms)', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => array(
					'ms' => array(
						'min'  => 1000,
						'max'  => 60000,
						'step' => 100,
					),
				),
				'default'   => array(
					'unit' => 'ms',
					'size' => 6000,
				),
				'condition' => array(
					'enable_autoplay' => 'yes',
				),
			)
		);

		/**
		 * Speed
		 */
		$this->add_control(
			'speed',
			array(
				'label'   => _x( 'Speed', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::SLIDER,
				'range'   => array(
					'ms' => array(
						'min'  => 100,
						'max'  => 10000,
						'step' => 100,
					),
				),
				'default' => array(
					'unit' => 'ms',
					'size' => 800,
				),
			)
		);

		/**
		 * Direction
		 */
		$this->add_control(
			'direction',
			array(
				'label'   => _x( 'Direction', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::CHOOSE,
				'default' => 'ltr',
				'options' => array(
					'ltr' => array(
						'title' => _x( 'Left to Right', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-chevron-double-right',
					),
					'rtl' => array(
						'title' => _x( 'Right to Left', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-chevron-double-left',
					),
				),
				'toggle'  => false,
			)
		);

		$this->end_controls_section();

	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		$classRow = 'justify-content-between';

		if ( ( $settings['enable_navigation'] && ! $settings['enable_progress'] ) || ( ! $settings['enable_navigation'] && $settings['enable_progress'] ) ) {
			$classRow = 'justify-content-center';
		}

		$this->add_render_attribute(
			'swiper',
			array(
				'class'                => array(
					'swiper',
					'swiper-container',
					'swiper-container-initialized',
					'slider',
					'slider-portfolio-item',
					'js-slider-portfolio-item',
				),
				'data-speed'           => $settings['speed']['size'],
				'dir'                  => $settings['direction'],
				'data-slides-per-view' => $settings['slides_per_view']['size'],
				'data-space-between'   => $settings['space_between']['size'],
				'data-centered-slides' => $settings['centered_slides'] ? 'true' : 'false',
			)
		);

		if ( array_key_exists( 'slides_per_view_tablet', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-slides-per-view-tablet', $settings['slides_per_view_tablet']['size'] );
		}

		if ( array_key_exists( 'slides_per_view_mobile', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-slides-per-view-mobile', $settings['slides_per_view_mobile']['size'] );
		}

		if ( array_key_exists( 'space_between_tablet', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-space-between-tablet', $settings['space_between_tablet']['size'] );
		}

		if ( array_key_exists( 'space_between_mobile', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-space-between-mobile', $settings['space_between_mobile']['size'] );
		}

		if ( array_key_exists( 'centered_slides_tablet', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-centered-slides-tablet', $settings['centered_slides_tablet'] ? 'true' : 'false' );
		}

		if ( array_key_exists( 'centered_slides_mobile', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-centered-slides-mobile', $settings['centered_slides_mobile'] ? 'true' : 'false' );
		}

		if ( $settings['enable_autoplay'] ) {
			$this->add_render_attribute(
				'swiper',
				array(
					'data-autoplay-enabled' => 'true',
					'data-autoplay-delay'   => is_array( $settings['autoplay_delay'] ) ? $settings['autoplay_delay']['size'] : 0,
				)
			);
		}
		?>
		<?php if ( ! empty( $settings['images_gallery'] ) ) : ?>
			<div <?php $this->print_render_attribute_string( 'swiper' ); ?>>
				<div class="swiper-wrapper">
					<?php if ( $settings['cover_to_slider'] && $settings['has_featured_image'] ) : ?>
						<div class="swiper-slide">
							<?php
								arr_the_lazy_image(
									array(
										'id'    => get_post_thumbnail_id(),
										'type'  => 'image',
										'class' => array(
											'wrapper' => array(),
											'image'   => array( 'swiper-lazy', 'of-cover' ),
										),
									)
								);
							?>
						</div>
					<?php endif; ?>
					<?php foreach ( $settings['images_gallery'] as $image ) : ?>
						<div class="swiper-slide">
							<?php
								arr_the_lazy_image(
									array(
										'id'    => $image['id'],
										'type'  => 'image',
										'class' => array(
											'wrapper' => array(),
											'image'   => array( 'swiper-lazy', 'of-cover' ),
										),
									)
								);
							?>
						</div>
					<?php endforeach; ?>
				</div>
				<?php if ( $settings['enable_navigation'] || $settings['enable_progress'] ) : ?>
					<div class="container">
						<div class="row align-items-center <?php echo $classRow; ?>">
							<?php if ( $settings['enable_navigation'] ) : ?>
								<!-- naivgation -->
								<div class="col-auto">
									<div class="slider__arrows slider-portfolio-item__arrows">
										<div class="slider__arrow js-slider-portfolio-item__arrow-left eicon-angle-left"></div>
										<div class="slider__arrow js-slider-portfolio-item__arrow-right eicon-angle-right"></div>
									</div>
								</div>
								<!-- - naivgation -->
							<?php endif; ?>
							<?php if ( $settings['enable_progress'] ) : ?>
								<!-- progress bar -->
								<div class="col-auto">
									<div class="slider__progress">
										<div class="swiper swiper-container swiper-container-initialized slider__counter slider__counter_current js-slider-portfolio-item-counter-current">
											<div class="swiper-wrapper"></div>
										</div>
										<div class="slider__progressbar js-slider-portfolio-item-progress">
											<div class="slider__progressbar-fill"></div>
										</div>
										<div class="slider__counter slider__counter_total js-slider-portfolio-item-counter-total">00</div>
									</div>
								</div>
								<!-- - progress bar -->
							<?php endif; ?>
						</div>
					</div>
				<?php endif; ?>
			</div>
		<?php endif; ?>
		<?php
	}
}
