<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Arrigo_Elementor_Google_Map_Widget extends Arr_Widget_Base {
	protected static $_instance;

	public function get_name() {
		return 'arrigo-widget-google-map';
	}

	public function get_title() {
		return _x( 'Google Map', 'Elementor Widget Title', 'arrigo' );
	}

	public function get_icon() {
		return 'eicon-plug';
	}

	public function get_categories() {
		return array( 'arrigo-static' );
	}

	public function get_script_depends() {
		$key = $this->get_gmap_api_key();

		if ( $key ) {
			wp_enqueue_script( 'googlemap', '//maps.googleapis.com/maps/api/js?callback=Function.prototype&key=' . $key, array(), null, true );
		}

		return array( 'googlemap' );
	}

	private function get_gmap_api_key() {
		$gmap = get_option( 'arr_gmap' );
		return ( isset( $gmap['key'] ) && ! empty( $gmap['key'] ) ) ? $gmap['key'] : false;
	}

	public function wpml_widgets_to_translate_filter( $widgets ) {
		$name  = $this->get_name();
		$title = $this->get_title();

		$widgets[ $name ] = array(
			'conditions'        => array( 'widgetType' => $name ),
			'fields'            => array(),
			'integration-class' => 'WPML_Arrigo_Elementor_Google_Map',
		);

		return $widgets;
	}

	public function add_wpml_support() {
		add_filter( 'wpml_elementor_widgets_to_translate', array( $this, 'wpml_widgets_to_translate_filter' ) );
	}

	protected function register_controls() {
		/**
		 * Section Markers
		 */
		$this->start_controls_section(
			'markers_section',
			array(
				'label' => _x( 'Markers', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			)
		);

		/**
		 * Check for API Key
		 */
		$this->add_control(
			'is_key',
			array(
				'type'    => \Elementor\Controls_Manager::HIDDEN,
				'default' => $this->get_gmap_api_key(),
			)
		);

		/**
		 * No Google Map API Key Info
		 */
		$this->add_control(
			'message_api',
			array(
				'type'            => \Elementor\Controls_Manager::RAW_HTML,
				'raw'             => sprintf(
					'%1$s <a href="https://developers.google.com/maps/documentation/javascript/tutorial#api-key-billing-errors" target="_blank">%2$s</a><br><br>%3$s <a href="%4$s" target="_blank">%5$s</a> %6$s',
					_x( 'To make this widget work properly you need to obtain', 'Elementor Widget', 'arrigo' ),
					_x( 'an API key from Google', 'Elementor Widget', 'arrigo' ),
					_x( 'If you already have a valid and set up API key please enter it in', 'Elementor Widget', 'arrigo' ),
					admin_url( 'options-general.php?page=arr-setting-gmap' ),
					_x( 'in WordPress admin panel.', 'Elementor Widget', 'arrigo' ),
					_x( 'and go back here.', 'Elementor Widget', 'arrigo' )
				),
				'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
				'condition'       => array(
					'is_key' => false,
				),
			)
		);

		$repeater = new \Elementor\Repeater();

		/**
		 * Marker Content
		 */
		$repeater->add_control(
			'marker_content',
			array(
				'label'       => _x( 'Content Box', 'Elementor Widget', 'arrigo' ),
				'description' => _x( 'That content will appear in a small window on marker click. You can place a helpful information here (e.g. company address).', 'Elementor Widget', 'arrigo' ),
				'type'        => \Elementor\Controls_Manager::TEXTAREA,
				'default'     => '',
			)
		);

		/**
		 * Marker Lat
		 */
		$repeater->add_control(
			'marker_lat',
			array(
				'label'   => _x( 'Latitude', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => '40.6700',
			)
		);

		/**
		 * Marker Long
		 */
		$repeater->add_control(
			'marker_lon',
			array(
				'label'   => _x( 'Longitude', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => '-73.9400',
			)
		);

		/**
		 * Marker Icon
		 */
		$repeater->add_control(
			'marker_icon',
			array(
				'label'       => _x( 'Icon', 'Elementor Widget', 'arrigo' ),
				'description' => _x( 'Upload a PNG image of a nice map pin.', 'Elementor Widget', 'arrigo' ),
				'type'        => \Elementor\Controls_Manager::MEDIA,
			)
		);

		/**
		 * Marker Icon Size
		 */

		$repeater->add_control(
			'marker_icon_size',
			array(
				'label'     => _x( 'Image Size', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::IMAGE_DIMENSIONS,
				'default'   => array(
					'width'  => '',
					'height' => '',
				),
				'condition' => array(
					'marker_icon!' => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		/**
		 * Markers Repeater Control
		 */
		$this->add_control(
			'markers',
			array(
				'type'          => \Elementor\Controls_Manager::REPEATER,
				'fields'        => $repeater->get_controls(),
				'title_field'   => '{{{ marker_content }}}',
				'prevent_empty' => true,
				'condition'     => array(
					'is_key!' => false,
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Section Style
		 */
		$this->start_controls_section(
			'style_section',
			array(
				'label' => _x( 'Style', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			)
		);

		/**
		 * Zoom
		 */
		$this->add_control(
			'map_zoom',
			array(
				'label'       => _x( 'Zoom Level', 'Elementor Widget', 'arrigo' ),
				'description' => _x( 'Applicable if there is only a single marker on the map', 'Elementor Widget', 'arrigo' ),
				'type'        => \Elementor\Controls_Manager::SLIDER,
				'range'       => array(
					'px' => array(
						'min' => 1,
						'max' => 20,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => 8,
				),
				'condition'   => array(
					'is_key!' => false,
				),
			)
		);

		/**
		 * Height
		 */
		$this->add_responsive_control(
			'map_height',
			array(
				'label'      => _x( 'Map Height', 'Elementor Widget', 'arrigo' ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 1440,
					),
					'vh' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'default'    => array(
					'size' => 600,
					'unit' => 'px',
				),
				'size_units' => array( 'px', 'vh' ),
				'selectors'  => array(
					'{{WRAPPER}} .gmap__container' => 'min-height: {{SIZE}}{{UNIT}};',
				),
			)
		);

		/**
		 * Snazzy Styles
		 */
		$this->add_control(
			'snazzy_styles',
			array(
				'label'       => _x( 'Snazzy Maps Styles', 'Elementor Widget', 'arrigo' ),
				'type'        => \Elementor\Controls_Manager::TEXTAREA,
				'description' => sprintf(
					'%1$s <a href="https://snazzymaps.com/explore" target="_blank">%2$s</a> %3$s',
					_x( 'You can preview and generate custom map styles on', 'Elementor Widget', 'arrigo' ),
					_x( 'Snazzy Maps.', 'Elementor Widget', 'arrigo' ),
					_x( 'Just copy-paste javascript style array in this field.', 'Elementor Widget', 'arrigo' )
				),
				'default'     => '[{"featureType":"water","elementType":"geometry","stylers":[{"color":"#e9e9e9"},{"lightness":17}]},{"featureType":"landscape","elementType":"geometry","stylers":[{"color":"#f5f5f5"},{"lightness":20}]},{"featureType":"road.highway","elementType":"geometry.fill","stylers":[{"color":"#ffffff"},{"lightness":17}]},{"featureType":"road.highway","elementType":"geometry.stroke","stylers":[{"color":"#ffffff"},{"lightness":29},{"weight":0.2}]},{"featureType":"road.arterial","elementType":"geometry","stylers":[{"color":"#ffffff"},{"lightness":18}]},{"featureType":"road.local","elementType":"geometry","stylers":[{"color":"#ffffff"},{"lightness":16}]},{"featureType":"poi","elementType":"geometry","stylers":[{"color":"#f5f5f5"},{"lightness":21}]},{"featureType":"poi.park","elementType":"geometry","stylers":[{"color":"#dedede"},{"lightness":21}]},{"elementType":"labels.text.stroke","stylers":[{"visibility":"on"},{"color":"#ffffff"},{"lightness":16}]},{"elementType":"labels.text.fill","stylers":[{"saturation":36},{"color":"#333333"},{"lightness":40}]},{"elementType":"labels.icon","stylers":[{"visibility":"off"}]},{"featureType":"transit","elementType":"geometry","stylers":[{"color":"#f2f2f2"},{"lightness":19}]},{"featureType":"administrative","elementType":"geometry.fill","stylers":[{"color":"#fefefe"},{"lightness":20}]},{"featureType":"administrative","elementType":"geometry.stroke","stylers":[{"color":"#fefefe"},{"lightness":17},{"weight":1.2}]}]',
			)
		);

		$this->end_controls_section();

	}

	protected function render() {

		$settings = $this->get_settings_for_display();

		$this->add_render_attribute(
			'gmap',
			array(
				'class'                   => 'gmap',
				'data-gmap-zoom'          => is_array( $settings['map_zoom'] ) ? $settings['map_zoom']['size'] : 8,
				'data-gmap-snazzy-styles' => preg_replace( '^\n|\r|\s+|\s+^', '', $settings['snazzy_styles'] ),
			)
		);

		?>

		<div <?php $this->print_render_attribute_string( 'gmap' ); ?>>
			<div class="gmap__container"></div>
			<?php if ( $settings['markers'] ) : ?>
				<?php foreach ( $settings['markers'] as $index => $item ) : ?>
					<?php
						$rowKey = $this->get_repeater_setting_key( 'marker', 'markers', $index );
						$this->add_render_attribute(
							$rowKey,
							array(
								'class'               => 'gmap__marker d-none',
								'data-marker-lat'     => $item['marker_lat'],
								'data-marker-lon'     => $item['marker_lon'],
								'data-marker-content' => $item['marker_content'],
								'data-marker-img'     => $item['marker_icon']['url'],
								'data-marker-width'   => $item['marker_icon_size']['width'],
								'data-marker-height'  => $item['marker_icon_size']['height'],
							)
						);
					?>
					<div <?php $this->print_render_attribute_string( $rowKey ); ?>></div>
				<?php endforeach; ?>
			<?php endif; ?>
		</div>

		<?php
	}

	protected function content_template() {
		?>

		<#
			view.addRenderAttribute(
				'gmap',
				{
					'class': 'gmap',
					'data-gmap-snazzy-styles': settings.snazzy_styles,
					'data-gmap-zoom': settings.map_zoom.size,
				}
			);
		#>
		<div {{{ view.getRenderAttributeString( 'gmap' ) }}}>
			<div class="gmap__container"></div>
			<# if (settings.markers.length) { #>
				<# _.each( settings.markers, function(item, index) { #>
					<#
						var rowKey = view.getRepeaterSettingKey( 'marker', 'markers', index );
						view.addRenderAttribute(
							rowKey,
							{
								'class': 'gmap__marker d-none',
								'data-marker-lat': item.marker_lat,
								'data-marker-lon': item.marker_lon,
								'data-marker-content': item.marker_content,
								'data-marker-img': item.marker_icon.url,
								'data-marker-width': item.marker_icon_size.width,
								'data-marker-height': item.marker_icon_size.height,
							}
						);
					#>
					<div {{{ view.getRenderAttributeString( rowKey ) }}}></div>
				<# }); #>
			<# } #>
		</div>

		<?php
	}
}
