<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Arrigo_Elementor_CTA_Widget extends Arr_Widget_Base {
	protected static $_instance;

	public function get_name() {
		return 'arrigo-widget-cta';
	}

	public function get_title() {
		return _x( 'Call To Action', 'Elementor Widget Title', 'arrigo' );
	}

	public function get_icon() {
		return 'eicon-plug';
	}

	public function get_categories() {
		return array( 'arrigo-static' );
	}

	public function wpml_widgets_to_translate_filter( $widgets ) {
		$name  = $this->get_name();
		$title = $this->get_title();

		$widgets[ $name ] = array(
			'conditions' => array( 'widgetType' => $name ),
			'fields'     => array(
				array(
					'field'       => 'heading',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, _x( 'Heading', 'Elementor Widget', 'arrigo' ) ),
					'editor_type' => 'LINE',
				),
				array(
					'field'       => 'subheading',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, _x( 'Subheading', 'Elementor Widget', 'arrigo' ) ),
					'editor_type' => 'LINE',
				),
				array(
					'field'       => 'button_title',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, _x( 'Button', 'Elementor Widget', 'arrigo' ) ),
					'editor_type' => 'LINE',
				),
				'link' => array(
					'field'       => 'button_link',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, _x( 'Button: Link', 'Elementor Widget', 'arrigo' ) ),
					'editor_type' => 'LINK',
				),
			),
		);

		return $widgets;
	}

	public function add_wpml_support() {
		add_filter( 'wpml_elementor_widgets_to_translate', array( $this, 'wpml_widgets_to_translate_filter' ) );
	}

	protected function register_controls() {

		/**
		 * Section Content
		 */
		$this->start_controls_section(
			'content_section',
			array(
				'label' => _x( 'Content', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			)
		);

		/**
		 * Heading
		 */
		$this->add_control(
			'heading',
			array(
				'label'       => _x( 'Heading', 'Elementor Widget', 'arrigo' ),
				'label_block' => true,
				'type'        => \Elementor\Controls_Manager::TEXT,
			)
		);

		/**
		 * Subheading
		 */
		$this->add_control(
			'subheading',
			array(
				'label'       => _x( 'Subheading', 'Elementor Widget', 'arrigo' ),
				'label_block' => true,
				'type'        => \Elementor\Controls_Manager::TEXT,
			)
		);

		$this->end_controls_section();

		/**
		 * Section Image
		 */
		$this->start_controls_section(
			'image_section',
			array(
				'label' => _x( 'Image', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			)
		);

		/**
		 * Image Src
		 */
		$this->add_control(
			'image',
			array(
				'label'   => _x( 'Choose Image', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::MEDIA,
				'default' => array(
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				),
			)
		);

		/**
		 * Image Fit
		 */
		$this->add_control(
			'image_fit',
			array(
				'label'     => _x( 'Image Fit Mode', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => array(
					'contain' => array(
						'title' => _x( 'Contain', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-image-before-after',
					),
					'cover'   => array(
						'title' => _x( 'Cover', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-image',
					),
				),
				'default'   => 'cover',
				'toggle'    => false,
				'condition' => array(
					'image!' => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		/**
		 * Image Parallax
		 */
		$this->add_control(
			'image_parallax',
			array(
				'label'     => _x( 'Enable parallax', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => array(
					'image!' => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		/**
		 * Image Parallax Speed
		 */
		$this->add_control(
			'image_parallax_speed',
			array(
				'label'     => _x( 'Parallax Speed', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => array(
					'factor' => array(
						'min'  => -1.0,
						'max'  => 2.0,
						'step' => 0.01,
					),
				),
				'default'   => array(
					'unit' => 'factor',
					'size' => 0.9,
				),
				'condition' => array(
					'image_parallax' => 'yes',
					'image!'         => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Section Button
		 */
		$this->start_controls_section(
			'button_section',
			array(
				'label' => _x( 'Button', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			)
		);

		/**
		 * Button Title
		 */
		$this->add_control(
			'button_title',
			array(
				'label'       => _x( 'Title', 'Elementor Widget', 'arrigo' ),
				'label_block' => true,
				'type'        => \Elementor\Controls_Manager::TEXT,
				'default'     => _x( 'Button', 'Elementor Widget', 'arrigo' ),
			)
		);

		/**
		 * Button URL
		 */
		$this->add_control(
			'button_link',
			array(
				'label'         => _x( 'Link', 'Elementor Widget', 'arrigo' ),
				'type'          => \Elementor\Controls_Manager::URL,
				'placeholder'   => _x( 'https://...', 'Elementor Widget', 'arrigo' ),
				'show_external' => true,
				'default'       => array(
					'url'         => '#',
					'is_external' => true,
					'nofollow'    => true,
				),
			)
		);

		/**
		 * Button Style
		 */
		$this->add_control(
			'button_style',
			array(
				'label'       => _x( 'Style', 'Elementor Widget', 'arrigo' ),
				'label_block' => true,
				'type'        => \Elementor\Controls_Manager::SELECT,
				'options'     => array(
					'button_bordered' => _x( 'Bordered', 'Elementor Widget', 'arrigo' ),
					'button_solid'    => _x( 'Solid', 'Elementor Widget', 'arrigo' ),
				),
				'default'     => 'button_solid',
			)
		);

		/**
		 * Button Color
		 */
		$this->add_control(
			'button_color',
			array(
				'label'       => _x( 'Color', 'Elementor Widget', 'arrigo' ),
				'label_block' => true,
				'type'        => \Elementor\Controls_Manager::SELECT,
				'options'     => array(
					'button_accent' => _x( 'Accent', 'Elementor Widget', 'arrigo' ),
					'button_dark'   => _x( 'Dark', 'Elementor Widget', 'arrigo' ),
					'button_light'  => _x( 'Light', 'Elementor Widget', 'arrigo' ),
					'button_black'  => _x( 'Black', 'Elementor Widget', 'arrigo' ),
					'button_white'  => _x( 'White', 'Elementor Widget', 'arrigo' ),
				),
				'default'     => 'button_accent',
			)
		);

		$this->end_controls_section();

		/**
		 * Section Layout
		 */
		$this->start_controls_section(
			'layout_section',
			array(
				'label' => _x( 'Layout', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_LAYOUT,
			)
		);

		/**
		 * Layout
		 */
		$this->add_control(
			'layout',
			array(
				'label'   => _x( 'Image Position', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::CHOOSE,
				'options' => array(
					'col-lg-6 offset-lg-6' => array(
						'title' => _x( 'Image Left', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-h-align-left',
					),
					'col'                  => array(
						'title' => _x( 'Image Fullwidth', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-h-align-stretch',
					),
					'col-lg-6'             => array(
						'title' => _x( 'Image Right', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-h-align-right',
					),
				),
				'default' => 'col-lg-6 offset-lg-6',
				'toggle'  => false,
			)
		);

		/**
		 * Overlay
		 */
		$this->add_responsive_control(
			'overlay',
			array(
				'label'          => _x( 'Image Overlay', 'Elementor Widget', 'arrigo' ),
				'label_block'    => true,
				'type'           => \Elementor\Controls_Manager::SELECT,
				'options'        => array(
					'initial' => _x( 'None', 'Elementor Widget', 'arrigo' ),
					'#181818' => _x( 'Dark', 'Elementor Widget', 'arrigo' ),
					'#ffffff' => _x( 'Light', 'Elementor Widget', 'arrigo' ),
				),
				'default'        => 'initial',
				'default_tablet' => '#ffffff',
				'default_mobile' => '#ffffff',
				'selectors'      => array(
					'{{WRAPPER}} .overlay' => 'background-color: {{overlay}};',
				),
			)
		);

		/**
		 * Overlay Opacity
		 */
		$this->add_control(
			'overlay_opacity',
			array(
				'label'     => __( 'Image Overlay Opacity', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'default'   => array(
					'size' => .8,
				),
				'range'     => array(
					'px' => array(
						'max'  => 1,
						'step' => 0.01,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .overlay' => 'opacity: {{SIZE}};',
				),
			)
		);

		/**
		 * Layout Content
		 */
		$this->add_control(
			'content_position',
			array(
				'label'     => _x( 'Content Position', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => array(
					'col-lg-6'                         => array(
						'title' => _x( 'Content Left', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-h-align-left',
					),
					'col-lg-8 offset-lg-2 text-center' => array(
						'title' => _x( 'Content Center', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-h-align-center',
					),
					'col-lg-6 offset-lg-6'             => array(
						'title' => _x( 'Content Right', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-h-align-right',
					),
				),
				'default'   => 'col-lg-6',
				'toggle'    => true,
				'condition' => array(
					'layout' => 'col',
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Section Style
		 */
		$this->start_controls_section(
			'style_section',
			array(
				'label' => _x( 'Style', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			)
		);

		/**
		 * Height
		 */
		$this->add_control(
			'height',
			array(
				'label'   => _x( 'Height', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'default' => 'default',
				'options' => array(
					'default'    => _x( 'Default', 'Elementor Widget', 'arrigo' ),
					'full'       => _x( 'Fit To Screen', 'Elementor Widget', 'arrigo' ),
					'min-height' => _x( 'Min Height', 'Elementor Widget', 'arrigo' ),
				),
			)
		);

		/**
		 * Min Height
		 */
		$this->add_responsive_control(
			'custom_height',
			array(
				'label'      => _x( 'Minimum Height', 'Elementor Widget', 'arrigo' ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'default'    => array(
					'size' => 400,
					'unit' => 'px',
				),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 1440,
					),
					'vh' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'size_units' => array( 'px', 'vh' ),
				'selectors'  => array(
					'{{WRAPPER}} .section__inner' => 'min-height: {{SIZE}}{{UNIT}};',
					// '{{WRAPPER}} .section__inner:after' => 'content: ""; min-height: inherit;', // Hack for IE11
				),
				'condition'  => array(
					'height' => array( 'min-height' ),
				),
			)
		);

		/**
		 * Inverse Text Color
		 */
		$this->add_control(
			'inverse_text_color',
			array(
				'label'        => _x( 'Inverse Text Color', 'Elementor Widget', 'arrigo' ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'default'      => 'no',
				'return_value' => 'color-white',
			)
		);

		/**
		 * Show Decoration Square
		 */
		$this->add_control(
			'show_decoration_square',
			array(
				'label'   => __( 'Show Square Decoration', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		/**
		 * Show Decoration Line
		 */
		$this->add_control(
			'show_decoration_line',
			array(
				'label'   => __( 'Show Diagonal Line Decoration', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();

		/**
		 * Section Animation
		 */
		$this->start_controls_section(
			'animation_section',
			array(
				'label' => _x( 'Animation', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			)
		);

		/**
		 * OS Animation
		 */
		$this->add_control(
			'enable_animation',
			array(
				'label'        => _x( 'Enable on-scroll animation', 'Elementor Widget', 'arrigo' ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'return_value' => 'data-os-animation="data-os-animation"',
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'heading' );
		$this->add_inline_editing_attributes( 'subheading' );
		$this->add_inline_editing_attributes( 'button_title' );

		$this->add_render_attribute( 'section', 'class', array( 'section', 'section-cta' ) );
		$this->add_render_attribute( 'section_inner', 'class', array( 'section__inner' ) );
		$this->add_render_attribute( 'image', 'class', array( 'col', 'section-cta__bg' ) );
		$this->add_render_attribute( 'content', 'class', array( 'col-lg-6', 'section-cta__content', $settings['layout'] ) );

		switch ( $settings['layout'] ) {
			case 'col-lg-6 offset-lg-6':
				$this->add_render_attribute( 'image', 'class', 'col-lg-6' );
				break;

			case 'col-lg-6':
				$this->add_render_attribute( 'image', 'class', array( 'col-lg-6', 'offset-lg-6' ) );
				break;

			case 'col':
				$this->add_render_attribute( 'content', 'class', $settings['content_position'], true );
				break;
		}

		if ( $settings['height'] == 'full' || $settings['height'] == 'min-height' ) {
			$this->add_render_attribute( 'section', 'class', array( 'section-fullheight' ) );
			$this->add_render_attribute( 'section_inner', 'class', array( 'section-fullheight__inner' ) );
		}

		if ( $settings['inverse_text_color'] ) {
			$this->add_render_attribute( 'section', 'class', $settings['inverse_text_color'] );
		}

		if ( $settings['image_parallax'] ) {
			$this->add_render_attribute(
				'image',
				array(
					'class'         => 'jarallax',
					'data-speed'    => is_array( $settings['image_parallax_speed'] ) ? $settings['image_parallax_speed']['size'] : 0,
					'data-img-size' => $settings['image_fit'],
				)
			);
		}

		if ( $settings['enable_animation'] ) {
			$this->add_render_attribute( 'section', 'data-os-animation' );
		}

		?>
		<div <?php $this->print_render_attribute_string( 'section' ); ?>>
			<div <?php $this->print_render_attribute_string( 'section_inner' ); ?>>
				<?php if ( ! empty( $settings['image']['id'] ) ) : ?>
					<div <?php $this->print_render_attribute_string( 'image' ); ?>>
						<?php
							arr_the_lazy_image(
								array(
									'id'    => $settings['image']['id'],
									'type'  => 'background',
									'class' => array(
										'wrapper' => false,
										'image'   => array( 'jarallax-img', 'of-' . $settings['image_fit'] ),
									),
								)
							);
						?>
					</div>
				<?php endif; ?>
				<div class="container">
					<div class="row align-items-center">
						<div <?php $this->print_render_attribute_string( 'content' ); ?>>
							<?php if ( ! empty( $settings['heading'] ) || ! empty( $settings['subheading'] ) ) : ?>
								<header class="section-cta__header">
									<?php if ( ! empty( $settings['heading'] ) ) : ?>
										<h2 <?php $this->print_render_attribute_string( 'heading' ); ?>><?php echo $settings['heading']; ?></h2>
									<?php endif; ?>
									<?php if ( ! empty( $settings['subheading'] ) ) : ?>
										<h4 <?php $this->print_render_attribute_string( 'subheading' ); ?>><?php echo $settings['subheading']; ?></h4>
									<?php endif; ?>
								</header>
							<?php endif; ?>
							<?php if ( ! empty( $settings['button_title'] ) ) : ?>
								<div class="section-cta__wrapper-button">
									<?php
										$this->add_render_attribute(
											'button',
											array(
												'class' => array( 'button', $settings['button_style'], $settings['button_color'] ),
												'href'  => $settings['button_link']['url'],
											)
										);

									if ( $settings['button_link']['is_external'] ) {
										$this->add_render_attribute( 'button', 'target', '_blank' );
									}

									if ( $settings['button_link']['nofollow'] ) {
										$this->add_render_attribute( 'button', 'rel', 'nofollow' );
									}
									?>
									<a <?php $this->print_render_attribute_string( 'button' ); ?>><span <?php $this->print_render_attribute_string( 'button_title' ); ?>><?php echo $settings['button_title']; ?></span></a>
								</div>
							<?php endif; ?>
						</div>
					</div>
				</div>
			</div>
			<?php if ( $settings['show_decoration_square'] ) : ?>
				<div class="section-cta__decor-el-1 d-none d-lg-block"></div>
			<?php endif; ?>
			<?php if ( $settings['show_decoration_line'] ) : ?>
				<div class="section-cta__decor-el-2 d-none d-lg-block"></div>
			<?php endif; ?>
			<div class="overlay"></div>
		</div>

		<?php
	}

	protected function content_template() {
		?>
			<#
				view.addInlineEditingAttributes( 'heading' );
				view.addInlineEditingAttributes( 'subheading' );
				view.addInlineEditingAttributes( 'button_title' );

				view.addRenderAttribute( 'section', 'class', [ 'section', 'section-cta' ] );
				view.addRenderAttribute( 'section_inner', 'class', [ 'section__inner' ] );
				view.addRenderAttribute( 'image', 'class', [ 'col', 'section-cta__bg' ] );
				view.addRenderAttribute( 'content', 'class', [ 'col-lg-6', 'section-cta__content', settings.layout ] );

				switch (settings.layout) {
					case 'col-lg-6 offset-lg-6':
						view.addRenderAttribute( 'image', 'class', 'col-lg-6' );
						break;
					case 'col-lg-6':
						view.addRenderAttribute( 'image', 'class', ['col-lg-6', 'offset-lg-6'] );
						break;
					case 'col':
						view.addRenderAttribute( 'content', 'class', settings.content_position, true );
						break;
				}

				if (settings.height == 'full' || settings.height == 'min-height') {
					view.addRenderAttribute( 'section', 'class', [ 'section-fullheight' ] );
					view.addRenderAttribute( 'section_inner', 'class', [ 'section-fullheight__inner' ] );	
				}

				if ( settings.inverse_text_color ) {
					view.addRenderAttribute( 'section', 'class', settings.inverse_text_color );
				}

				if (settings.image_parallax) {
					view.addRenderAttribute(
						'image', {
							'class': 'jarallax',
							'data-speed': settings.image_parallax_speed.size,
							'data-img-size': settings.image_fit
						}
					);
					view.addRenderAttribute( 'img', 'class', 'jarallax-img' );
				}
			#>
			<div {{{ view.getRenderAttributeString( 'section' ) }}}>
				<# if (settings.image.url) { #>
					<div {{{ view.getRenderAttributeString( 'image' ) }}}>
						<#
							view.addRenderAttribute(
								'img', {
									'class': 'of-' + settings.image_fit,
									'src': settings.image.url,
									'alt': ''
								}
							);
						#>
						<img {{{ view.getRenderAttributeString( 'img' ) }}}/>
					</div>
				<# } #>
				<div {{{ view.getRenderAttributeString( 'section_inner' ) }}}>
					<div class="container">
						<div class="row align-items-center">
							<div {{{ view.getRenderAttributeString( 'content' ) }}}>
								<# if (settings.heading || settings.subheading) { #>
									<header class="section-cta__header">
										<# if (settings.heading) { #>
											<h2 {{{ view.getRenderAttributeString( 'heading' ) }}}>{{{ settings.heading }}}</h2>
										<# } #>
										<# if (settings.subheading) { #>
											<h4 {{{ view.getRenderAttributeString( 'subheading' ) }}}>{{{ settings.subheading }}}</h4>
										<# } #>
									</header>
								<# } #>
								<# if (settings.button_title) { #>
									<#
										view.addRenderAttribute(
											'button', {
												'class': [ 'button', settings.button_style, settings.button_color ],
												'href': settings.button_link.url,
											}
										);

										if ( settings.button_link.is_external ) {
											view.addRenderAttribute( 'button', 'target', '_blank' );
										}

										if ( settings.button_link.nofollow ) {
											view.addRenderAttribute( 'button', 'rel', 'nofollow' );
										}
									#>
									<div class="section-cta__wrapper-button">
										<a {{{ view.getRenderAttributeString( 'button' ) }}}><span {{{ view.getRenderAttributeString( 'button_title' ) }}}>{{{ settings.button_title }}}</span></a>
									</div>
								<# } #>
							</div>
						</div>
					</div>
				</div>
				<# if (settings.show_decoration_square) { #>
					<div class="section-cta__decor-el-1 d-none d-lg-block"></div>
				<# } #>
				<# if (settings.show_decoration_line) { #>
					<div class="section-cta__decor-el-2 d-none d-lg-block"></div>
				<# } #>
				<div class="overlay"></div>
			</div>
		<?php
	}
}
