<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Arrigo_Elementor_Featured_Image_Widget extends Arr_Widget_Base {
	protected static $_instance;

	private $featured_image;

	public function get_name() {
		return 'arrigo-widget-featured-image';
	}

	public function get_title() {
		return _x( 'Featured Image', 'Elementor Widget Title', 'arrigo' );
	}

	public function get_icon() {
		return 'eicon-sitemap';
	}

	public function get_categories() {
		return array( 'arrigo-dynamic' );
	}

	protected function is_dynamic_content(): bool {
		return true;
	}

	protected function register_controls() {
		if ( has_post_thumbnail() ) {
			$this->featured_image = get_post_thumbnail_id();
		} else {
			$this->featured_image = false;
		}

		/**
		 * Section Image
		 */
		$this->start_controls_section(
			'image_section',
			array(
				'label' => _x( 'Image', 'Elementor Widget Section', 'arrigo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			)
		);

		/**
		 * Check for Featured Image
		 */
		$this->add_control(
			'has_featured_image',
			array(
				'type'    => \Elementor\Controls_Manager::HIDDEN,
				'default' => $this->featured_image,
			)
		);

		/**
		 * No featured Image
		 */
		$this->add_control(
			'no_featured_image_info',
			array(
				'type'            => \Elementor\Controls_Manager::RAW_HTML,
				'raw'             => sprintf(
					'%1$s',
					_x( 'In order to use this widget please make sure you have Featured Image for set for the current page.', 'Elementor Widget', 'arrigo' )
				),
				'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
				'condition'       => array(
					'has_featured_image' => '',
				),
			)
		);

		/**
		 * Image Parallax
		 */
		$this->add_control(
			'image_parallax',
			array(
				'label'     => _x( 'Enable parallax', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => '',
				'condition' => array(
					'has_featured_image!' => '',
				),
			)
		);

		/**
		 * Image Fit
		 */
		$this->add_control(
			'image_fit',
			array(
				'label'     => _x( 'Image Fit Mode', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => array(
					'contain' => array(
						'title' => _x( 'Contain', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-image-before-after',
					),
					'cover'   => array(
						'title' => _x( 'Cover', 'Elementor Widget', 'arrigo' ),
						'icon'  => 'eicon-image',
					),
				),
				'default'   => 'cover',
				'toggle'    => false,
				'condition' => array(
					'image_parallax'      => 'yes',
					'has_featured_image!' => '',
				),
			)
		);

		/**
		 * Min Height
		 */
		$this->add_responsive_control(
			'min_height',
			array(
				'label'      => _x( 'Minimum Height', 'Elementor Widget', 'arrigo' ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'default'    => array(
					'size' => 400,
					'unit' => 'px',
				),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 1440,
					),
					'vh' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'size_units' => array( 'px', 'vh' ),
				'selectors'  => array(
					'{{WRAPPER}} .parallax-background' => 'min-height: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .parallax-background:after' => 'content: ""; min-height: inherit;', // Hack for IE11
				),
				'condition'  => array(
					'image_parallax'      => 'yes',
					'has_featured_image!' => '',
				),
			)
		);

		/**
		 * Image Parallax Speed
		 */
		$this->add_control(
			'image_parallax_speed',
			array(
				'label'     => _x( 'Parallax Speed', 'Elementor Widget', 'arrigo' ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => array(
					'factor' => array(
						'min'  => -1.0,
						'max'  => 2.0,
						'step' => 0.01,
					),
				),
				'default'   => array(
					'unit' => 'factor',
					'size' => 0.9,
				),
				'condition' => array(
					'image_parallax'      => 'yes',
					'has_featured_image!' => '',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		if ( has_post_thumbnail() ) {
			$this->featured_image = get_post_thumbnail_id();
		} else {
			$this->featured_image = false;
		}

		$this->add_render_attribute( 'image', 'class', 'parallax-background' );

		if ( $settings['image_parallax'] ) {

			$this->add_render_attribute(
				'image',
				array(
					'class'         => 'jarallax',
					'data-speed'    => $settings['image_parallax_speed']['size'],
					'data-img-size' => $settings['image_fit'],
				)
			);
		}
		?>
		<?php if ( $this->featured_image ) : ?>
			<?php if ( $settings['image_parallax'] ) : ?>
				<div <?php $this->print_render_attribute_string( 'image' ); ?>>
					<?php
						arr_the_lazy_image(
							array(
								'id'    => $this->featured_image,
								'type'  => 'image',
								'class' => array(
									'wrapper' => false,
									'image'   => array( 'jarallax-img', 'of-' . $settings['image_fit'] ),
								),
							)
						);
					?>
				</div>
			<?php else : ?>
				<?php
					arr_the_lazy_image(
						array(
							'id'   => $this->featured_image,
							'type' => 'image',
						)
					);
				?>
			<?php endif; ?>
		<?php endif; ?>
		<?php
	}
}
