<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Arr_Widget_Base extends Widget_Base {
	protected static $_instance;
	protected static $_posts;
	protected static $_post_type;
	protected static $_data_static_fields;

	public static function instance() {
		if ( is_null( static::$_instance ) ) {
			static::$_instance = new static();
		}

		return static::$_instance;
	}

	public function get_name() {}

	public function get_title() {}

	public function get_icon() {
		return 'eicon-plug';
	}

	public function get_categories() {
		return array( 'arrigo-static' );
	}

	protected function is_dynamic_content(): bool {
		return false;
	}

	public function has_widget_inner_wrapper(): bool {
		return ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
	}

	/**
	 * WordPress DB query for the posts
	 *
	 * @return void
	 */
	protected static function _get_posts() {
		// get class name in lowercase
		$class_name = ( new \ReflectionClass( static::class ) )->getShortName();
		$class_name = strtolower( $class_name );

		// filter to change current widget post type
		$args = apply_filters(
			'arr/elementor/' . $class_name . '/query_args',
			array(
				'post_type'      => static::$_post_type,
				'posts_per_page' => -1,
			)
		);

		// global query filter (affects all dynamic widgets)
		$args               = apply_filters( 'arr/elementor/dynamic_widgets/query_args', $args );
		static::$_post_type = $args['post_type'];

		$posts      = array();
		$taxonomies = array();
		$counter    = 0;

		$loop = new \WP_Query( $args );

		if ( $loop->have_posts() ) {
			while ( $loop->have_posts() ) {
				$loop->the_post();

				$post_id = get_the_ID();

				$posts[ $counter ]['id']               = $post_id;
				$posts[ $counter ]['title']            = get_the_title( $post_id );
				$posts[ $counter ]['permalink']        = get_the_permalink( $post_id );
				$posts[ $counter ]['image_id']         = get_post_thumbnail_id( $post_id );
				$posts[ $counter ]['image_url']        = get_the_post_thumbnail_url( $post_id, 'full' );
				$post_categories                       = get_the_terms( $post_id, 'arr_portfolio_category' );
				$description                           = '';
				$posts[ $counter ]['categories']       = $post_categories;
				$posts[ $counter ]['categories_names'] = array();

				if ( static::$_post_type === 'arr_services' ) {
					$description = get_post_meta( $post_id, 'service_short_description', true );
				} elseif ( static::$_post_type === 'arr_portfolio' ) {
					$description = get_post_meta( $post_id, 'portfolio_item_short_description', true );
				}

				$posts[ $counter ]['description'] = $description;
				$posts[ $counter ]['date']        = get_the_date( '', $post_id );

				if ( is_array( $post_categories ) ) {
					foreach ( $post_categories as $item ) {

						$arr = array(
							'slug' => $item->slug,
							'name' => $item->name,
						);

						array_push( $posts[ $counter ]['categories_names'], $item->name );

						// don't add the same item multiple times
						if ( ! in_array( $arr, $taxonomies ) ) {
							array_push( $taxonomies, $arr );
						}
					}
				}

				$counter++;
			}

			wp_reset_postdata();
		}

		static::$_posts = $posts;

		return static::$_posts;
	}

	/**
	 * Get all posts by a pre-set type
	 *
	 * @return array
	 */
	public static function get_posts() {
		if ( is_null( static::$_posts ) ) {
			static::_get_posts();
		}

		return static::$_posts;
	}

	/**
	 * Filter out disabled posts
	 *
	 * @return array
	 */
	public function get_posts_to_display() {
		$posts    = $this->get_posts();
		$settings = $this->get_settings_for_display();

		// static data source
		if ( array_key_exists( 'posts_data_source', $settings ) && $settings['posts_data_source'] === 'static' ) {

			if ( is_array( $settings['posts_static'] ) && ! empty( $settings['posts_static'] ) ) {
				foreach ( $settings['posts_static'] as $key => $value ) {

					$settings['posts_static'][ $key ]['id'] = $key;

					if ( array_key_exists( 'permalink', $settings['posts_static'][ $key ] ) ) {
						$settings['posts_static'][ $key ]['permalink_is_external'] = array_key_exists( 'is_external', $settings['posts_static'][ $key ]['permalink'] ) && ! ! $settings['posts_static'][ $key ]['permalink']['is_external'];
						$settings['posts_static'][ $key ]['permalink_nofollow']    = array_key_exists( 'nofollow', $settings['posts_static'][ $key ]['permalink'] ) && ! ! $settings['posts_static'][ $key ]['permalink']['nofollow'];
						$settings['posts_static'][ $key ]['permalink']             = $settings['posts_static'][ $key ]['permalink']['url'];
					}

					if ( array_key_exists( 'image', $settings['posts_static'][ $key ] ) ) {
						$settings['posts_static'][ $key ]['image_id']           = $settings['posts_static'][ $key ]['image']['id'];
						$settings['posts_static'][ $key ]['image_secondary_id'] = null;
					}

					if ( array_key_exists( 'video', $settings['posts_static'][ $key ] ) ) {
						$settings['posts_static'][ $key ]['video'] = $settings['posts_static'][ $key ]['video']['url'];
					}

					if ( array_key_exists( 'categories_names', $settings['posts_static'][ $key ] ) ) {
						$category_name                                        = $settings['posts_static'][ $key ]['categories_names'];
						$settings['posts_static'][ $key ]['categories_names'] = array( $category_name );
						$settings['posts_static'][ $key ]['categories'][0]    = (object) array(
							'name' => $category_name,
							'slug' => sanitize_title( $category_name ),
						);
					}

					if ( array_key_exists( 'subheading', $settings['posts_static'][ $key ] ) ) {
						$settings['posts_static'][ $key ]['subheading'] = $settings['posts_static'][ $key ]['subheading'];
					}
				}
			}

			$posts = $settings['posts_static'];

		} else { // dynamic data source

			// limit posts amount
			if ( isset( $settings['posts_amount']['size'] ) && $settings['posts_amount']['size'] > 0 ) {
				array_splice( $posts, $settings['posts_amount']['size'] );
			}

			// only "enabled" posts
			$posts = array_filter(
				$posts,
				function( $item ) {
					$settings = $this->get_settings_for_display();
					return ( array_key_exists( 'enabled' . $item['id'], $settings ) ) && ( $settings[ 'enabled' . $item['id'] ] );
				}
			);

		}

		return $posts;
	}

	protected function add_controls_posts_toggles( $extra_controls = array() ) {
		$posts = $this->get_posts();

		$this->add_controls_static_fields();

		if ( ! $posts ) {
			return;
		}

		$post_type     = static::$_post_type;
		$post_type_obj = get_post_type_object( $post_type );

		/**
		 * Not Editable Widget Info
		 */
		$this->add_control(
			'dynamic_content_info',
			array(
				'type'            => \Elementor\Controls_Manager::RAW_HTML,
				'raw'             => sprintf(
					'%1$s <strong>%2$s.</strong> %3$s<br><br>%4$s <a href="%5$s" target="_blank">%6$s</a>',
					esc_html__( 'This widget displays content dynamically from the existing', 'arrigo' ),
					$post_type_obj->labels->name,
					esc_html__( 'It\'s not editable directly through Elementor Page Builder.', 'arrigo' ),
					esc_html__( 'You can edit or re-order your posts', 'arrigo' ),
					admin_url( 'edit.php?post_type=' . $post_type ),
					esc_html__( 'in WordPress admin panel', 'arrigo' )
				),
				'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
			)
		);

		$this->add_control(
			'posts_amount',
			array(
				'label'   => _x( 'Number of Posts to Display (0 for all)', 'Elementor Widget', 'arrigo' ),
				'type'    => \Elementor\Controls_Manager::SLIDER,
				'range'   => array(
					'number' => array(
						'min'  => 0,
						'max'  => 16,
						'step' => 1,
					),
				),
				'default' => array(
					'unit' => 'number',
					'size' => 4,
				),
			)
		);

		foreach ( $posts as $index => $item ) {
			/**
			 * Heading Item
			 */
			$id = 'heading_item' . $item['id'];
			$this->add_control(
				$id,
				array(
					'raw'        => sprintf(
						'<h3 class="elementor-control-title"><strong>%1$s</strong>&nbsp;&nbsp;<a href="%2$s" target="_blank"><i class="eicon-edit"></i></a></h3>',
						$item['title'],
						admin_url( 'post.php?post=' . $item['id'] . '&action=edit' ),
						esc_html__( 'Edit', 'arrigo' )
					),
					'type'       => \Elementor\Controls_Manager::RAW_HTML,
					'separator'  => 'before',
					'conditions' => array(
						'relation' => 'and',
						'terms'    => array(
							array(
								'relation' => 'or',
								'terms'    => array(
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '>',
										'value'    => $index,
									),
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '<=',
										'value'    => '0',
									),
								),
							),
							array(
								'relation' => 'and',
								'terms'    => array(
									array(
										'name'     => 'posts_data_source',
										'operator' => '===',
										'value'    => 'dynamic',
									),
								),
							),
						),
					),
				)
			);

			if ( is_array( $extra_controls ) && ! empty( $extra_controls ) ) {
				/**
				 * Image Orientation
				 */
				if ( in_array( 'orientation', $extra_controls ) ) {
					$id = 'orientation' . $item['id'];
					$this->add_control(
						$id,
						array(
							'label'      => _x( 'Image Orientation', 'Elementor Widget', 'arrigo' ),
							'type'       => \Elementor\Controls_Manager::SELECT,
							'conditions' => array(
								'relation' => 'or',
								'terms'    => array(
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '>',
										'value'    => $index,
									),
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '<=',
										'value'    => '0',
									),
								),
							),
							'options'    => array(
								'figure-portfolio_horizontal' => _x( 'Landscape', 'Elementor Widget', 'arrigo' ),
								'figure-portfolio_vertical'   => _x( 'Portrait', 'Elementor Widget', 'arrigo' ),
							),
							'default'    => 'figure-portfolio_horizontal',
						)
					);
				}

				/**
				 * Content Alignment
				 */
				if ( in_array( 'position', $extra_controls ) ) {
					$id = 'content' . $item['id'];
					$this->add_control(
						$id,
						array(
							'label'      => _x( 'Image Position', 'Elementor Widget', 'arrigo' ),
							'type'       => \Elementor\Controls_Manager::SELECT,
							'conditions' => array(
								'relation' => 'or',
								'terms'    => array(
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '>',
										'value'    => $index,
									),
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '<=',
										'value'    => '0',
									),
								),
							),
							'options'    => array(
								'figure-portfolio_left'  => _x( 'Image Left', 'Elementor Widget', 'arrigo' ),
								'figure-portfolio_right' => _x( 'Image Right', 'Elementor Widget', 'arrigo' ),
							),
							'default'    => 'figure-portfolio_left',
						)
					);
				}

				/**
				 * Icon
				 */
				if ( in_array( 'icon', $extra_controls ) ) {
					$id = 'icon' . $item['id'];
					$this->add_control(
						$id,
						array(
							'label'      => _x( 'Icon', 'Elementor Widget', 'arrigo' ),
							'type'       => \Elementor\Controls_Manager::ICON,
							'conditions' => array(
								'relation' => 'or',
								'terms'    => array(
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '>',
										'value'    => $index,
									),
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '<=',
										'value'    => '0',
									),
								),
							),
							'separator'  => 'after',
						)
					);
				}

				/**
				 * Counter Value
				 */
				if ( in_array( 'counter', $extra_controls ) ) {
					$id = 'counter' . $item['id'];
					$this->add_control(
						$id,
						array(
							'label'      => _x( 'Counter', 'Elementor Widget', 'arrigo' ),
							'type'       => \Elementor\Controls_Manager::TEXT,
							'separator'  => 'after',
							'conditions' => array(
								'relation' => 'or',
								'terms'    => array(
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '>',
										'value'    => $index,
									),
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '<=',
										'value'    => '0',
									),
								),
							),
						)
					);
				}
			}

			/**
			 * Toggle
			 */
			$id = 'enabled' . $item['id'];
			$this->add_control(
				$id,
				array(
					'label'      => _x( 'Enabled', 'Elementor Widget', 'arrigo' ),
					'type'       => \Elementor\Controls_Manager::SWITCHER,
					'default'    => 'yes',
					'conditions' => array(
						'relation' => 'and',
						'terms'    => array(
							array(
								'relation' => 'or',
								'terms'    => array(
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '>',
										'value'    => $index,
									),
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '<=',
										'value'    => '0',
									),
								),
							),
							array(
								'relation' => 'and',
								'terms'    => array(
									array(
										'name'     => 'posts_data_source',
										'operator' => '===',
										'value'    => 'dynamic',
									),
								),
							),
						),
					),
				)
			);
		}
	}

	protected function add_controls_static_fields() {
		$static_fields = static::$_data_static_fields;

		if ( ! is_array( $static_fields ) || empty( $static_fields ) ) {
			$this->add_control(
				'posts_data_source',
				array(
					'label'   => esc_html__( 'Data Source Type', 'arrigo' ),
					'type'    => Controls_Manager::HIDDEN,
					'default' => 'dynamic',
				)
			);

			return;
		}
	}

	/**
	 * Collect the posts categories, remove duplicates
	 *
	 * @return array
	 */
	public function get_taxonomies_to_display( $posts = array(), $taxonomy = 'arr_portfolio_category' ) {
		$active_taxonomies = array();
		$hicpo_options     = get_option( 'hicpo_options' );
		$is_sortable       = false;
		$amounts           = array();
		$item_terms_set    = 'categories';

		// re-map taxonomy
		if ( static::$_post_type === 'post' ) {
			$taxonomy = 'category';
		}

		// sortable taxonomy via Intuitive Custom Posts Order plugin
		if ( class_exists( 'Hicpo' ) && is_array( $hicpo_options ) && array_key_exists( 'tags', $hicpo_options ) && ! empty( $hicpo_options['tags'] ) ) {
			$hicpo_tags  = $hicpo_options['tags'];
			$is_sortable = in_array( $taxonomy, $hicpo_tags );
		}

		foreach ( $posts as $item ) {
			if ( is_array( $item[ $item_terms_set ] ) ) {
				foreach ( $item[ $item_terms_set ] as $taxonomy ) {
					$arr = array(
						'slug' => $taxonomy->slug,
						'name' => $taxonomy->name,
					);

					// item already exists, increase counter;
					if ( in_array( $arr, $active_taxonomies ) ) {
						$amounts[ $taxonomy->slug ]++;
					} else { // don't add the same item multiple times
						$amounts[ $taxonomy->slug ] = 1;

						if ( $is_sortable ) {
							$active_taxonomies[ $taxonomy->term_order ] = $arr;
						} else {
							array_push( $active_taxonomies, $arr );
						}
					}
				}
			}
		}

		if ( $is_sortable ) {
			ksort( $active_taxonomies );
		}

		return array(
			'items'   => $active_taxonomies,
			'amounts' => $amounts,
		);
	}

	/**
	 * Translate the string values to the correct columns proportion value
	 */
	public function translate_columns_settings( $option ) {
		if ( ! $option ) {
			return 12;
		}

		if ( $option == '2dot4' ) {
			return 2.4;
		}

		return $option;
	}

	public function translate_columns_settings_inverse( $option ) {
		if ( ! $option ) {
			return 12;
		}

		if ( $option == 2.4 ) {
			return '2dot4';
		}

		return $option;
	}
}
