<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Arrigo Elementor Extension
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Arrigo_Elementor_Extension {

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '3.25.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.3';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Arrigo_Elementor_Extension The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Arrigo_Elementor_Extension An instance of the class.
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', array( $this, 'init' ) );
	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {
		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_missing_main_plugin' ) );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_minimum_elementor_version' ) );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_minimum_php_version' ) );
			return;
		}

		// Add Plugin actions
		add_action( 'elementor/elements/categories_registered', array( $this, 'add_elementor_widget_categories' ) );
		add_action( 'elementor/widgets/register', array( $this, 'init_widgets' ) );

		// Add WPML's Translation Management Support
		if ( class_exists( 'SitePress' ) ) {
			add_action( 'init', array( $this, 'add_widgets_wpml_support' ) );
		}
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'arrigo' ),
			'<strong>' . esc_html__( 'Arrigo Elementor Extension', 'arrigo' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'arrigo' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'arrigo' ),
			'<strong>' . esc_html__( 'Arrigo Elementor Extension', 'arrigo' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'arrigo' ) . '</strong>',
			self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'arrigo' ),
			'<strong>' . esc_html__( 'Arrigo Elementor Extension', 'arrigo' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'arrigo' ) . '</strong>',
			self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-error is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Register Custom Widget Categories
	 *
	 * @return void
	 */
	public function add_elementor_widget_categories( $elements_manager ) {
		$elements_manager->add_category(
			'arrigo-static',
			array(
				'title' => __( 'Arrigo Static Widgets', 'arrigo' ),
				'icon'  => 'eicon-plug',
			)
		);

		$elements_manager->add_category(
			'arrigo-dynamic',
			array(
				'title' => __( 'Arrigo Dynamic Widgets', 'arrigo' ),
				'icon'  => 'eicon-sitemap',
			)
		);
	}

		/**
		 * Include Widgets files
		 *
		 * Load widgets files
		 *
		 * @since 1.0.0
		 * @access private
		 */
	private function include_widgets_files() {
		// Custom Base Widget
		require_once __DIR__ . '/widgets/arr-widget-base.php';

		// Dynamic Widgets Require
		require_once __DIR__ . '/widgets/dynamic/featured-image-widget.php';
		require_once __DIR__ . '/widgets/dynamic/latest-posts-widget.php';
		require_once __DIR__ . '/widgets/dynamic/portfolio-slider-fullscreen-1-widget.php';
		require_once __DIR__ . '/widgets/dynamic/portfolio-slider-fullscreen-2-widget.php';
		require_once __DIR__ . '/widgets/dynamic/portfolio-grid-irregular-widget.php';
		require_once __DIR__ . '/widgets/dynamic/portfolio-masonry-grid-widget.php';
		require_once __DIR__ . '/widgets/dynamic/services-grid-widget.php';
		require_once __DIR__ . '/widgets/dynamic/services-slider-widget.php';

		// Static Widgets Require
		require_once __DIR__ . '/widgets/static/contact-widget.php';
		require_once __DIR__ . '/widgets/static/cta-widget.php';
		require_once __DIR__ . '/widgets/static/feature-widget.php';
		require_once __DIR__ . '/widgets/static/google-map-widget.php';
		require_once __DIR__ . '/widgets/static/header-section-widget.php';
		require_once __DIR__ . '/widgets/static/images-slider-widget.php';
		require_once __DIR__ . '/widgets/static/masthead-intro-widget.php';
		require_once __DIR__ . '/widgets/static/masonry-grid-widget.php';
		require_once __DIR__ . '/widgets/static/logo-client-widget.php';
		require_once __DIR__ . '/widgets/static/parallax-background-widget.php';
		require_once __DIR__ . '/widgets/static/pricing-table-widget.php';
		require_once __DIR__ . '/widgets/static/project-properties-widget.php';
		require_once __DIR__ . '/widgets/static/step-widget.php';
		require_once __DIR__ . '/widgets/static/team-member-widget.php';
		require_once __DIR__ . '/widgets/static/testimonials-slider-widget.php';
	}

	/**
	 * WPML compatibility classes
	 *
	 * Include files with WPML compatability classes
	 * for widgets with repeaters fields
	 *
	 * @since 1.0.0
	 * @access private
	 */
	private function include_wpml_files() {
		require_once __DIR__ . '/widgets/compatibility/wpml-google-map-widget.php';
		require_once __DIR__ . '/widgets/compatibility/wpml-pricing-table-widget.php';
		require_once __DIR__ . '/widgets/compatibility/wpml-project-properties-widget.php';
		require_once __DIR__ . '/widgets/compatibility/wpml-testimonials-slider-widget.php';
	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets() {
		$this->include_widgets_files();

		// Dynamic Widgets Init
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Featured_Image_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Latest_Posts_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Portfolio_Grid_Irregular_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Portfolio_Masonry_Grid_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Portfolio_Slider_Fullscreen_1_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Portfolio_Slider_Fullscreen_2_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Services_Grid_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Services_Slider_Widget::instance() );

		// Static Widgets Init
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Contact_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_CTA_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Feature_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Google_Map_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Header_Section_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Images_Slider_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Masthead_Intro_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Masonry_Grid_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Logo_Client_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Parallax_Background_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Pricing_Table_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Project_Properties_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Step_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Team_Member_Widget::instance() );
		\Elementor\Plugin::instance()->widgets_manager->register( \Elementor\Arrigo_Elementor_Testimonials_Slider_Widget::instance() );
	}

	public function add_widgets_wpml_support() {
		$this->include_wpml_files();
		$this->include_widgets_files();

		\Elementor\Arrigo_Elementor_Contact_Widget::instance()->add_wpml_support();
		\Elementor\Arrigo_Elementor_CTA_Widget::instance()->add_wpml_support();
		\Elementor\Arrigo_Elementor_Feature_Widget::instance()->add_wpml_support();
		\Elementor\Arrigo_Elementor_Google_Map_Widget::instance()->add_wpml_support();
		\Elementor\Arrigo_Elementor_Header_Section_Widget::instance()->add_wpml_support();
		\Elementor\Arrigo_Elementor_Masthead_Intro_Widget::instance()->add_wpml_support();
		\Elementor\Arrigo_Elementor_Pricing_Table_Widget::instance()->add_wpml_support();
		\Elementor\Arrigo_Elementor_Project_Properties_Widget::instance()->add_wpml_support();
		\Elementor\Arrigo_Elementor_Step_Widget::instance()->add_wpml_support();
		\Elementor\Arrigo_Elementor_Team_Member_Widget::instance()->add_wpml_support();
		\Elementor\Arrigo_Elementor_Testimonials_Slider_Widget::instance()->add_wpml_support();
	}

}
Arrigo_Elementor_Extension::instance();
